import {world, World, Entity} from "@minecraft/server";
import { report } from "../Infrastructure/ConstantLoader";

/**
 * A simple Database (I guess...)(First time)
 */
export class Database{
    constructor(identifier){
        /**
         * This Database's unique identifier
         */
        Object.defineProperty(this, "id", {value:identifier, writable: false});
    }
    /**
     * Sets a new table
     * @param {string} name The new table's key
     * @returns {Map}
     */
    newTable(name){
        if(name == ""){
            report("New Table name cannot be \"\"! Return undefined"); return undefined;
        }else {
            if(this.table.has(name)){
                report("New Table's name already used! Returns undefined"); return undefined;
            }else {
                this.table.set(name, new Map());
                report("New Table succesfully set! Returns "+name); return this.table.get(name);
            };
        };
    };
    /**
     * Gets a Table
     * @param {string} name The table's name to get
     */
    getTable(name){
        if(name==""){
            report("Table name not allowed! Returns undefined"); return undefined;
        }else {
            if(this.table.has(name)){
                report("Table found! Returns table "+name); return this.table.get(name);
            }else {
                report("Table not found! Returns undefined"); return undefined;
            };
        };
    };
    /**
     * Sets a value in a given table
     * @param {string} name Table name (key)
     * @param {string} key Table's value (key) to set
     * @param {string|number|Array|any} value The value to set to
     * @returns {undefined|string|number|Array|any}
     */
    setInTable(name, key, value){
        if(name == "" || key == "" || value == ""){
            report("Given \"\"! Returns undefined"); return undefined;
        }else {
            if(this.table.has(name)){
                if(this.table.get(name).has(key)){
                    report("Table value set! Returns value"); this.table.get(name).set(key, value); return this.table.get(name).get(key);
                }else {
                    report("Table value not found! Returns undefined"); return undefined;
                };
            }else {
                report("Table not found! Returns undefined"); return undefined;
            };
        };
    };
    /**
     * Sets a value in a given table
     * @param {string} name Table name (key)
     * @param {string} key Table's value (key) to set
     * @param {string|number|Array|any} value The value to set to
     * @returns {undefined|string|number|Array|any}
     */
    newKeyInTable(name, key, value){
        if(name == "" || key == ""){
            report("Given \"\"! Returns undefined"); return undefined;
        }else {
            if(this.table.has(name)){
                if(this.table.get(name).has(key)){
                    report("Table value already set! Returns table");return this.table.get(name).get(key);
                }else {
                    this.table.get(name).set(key, value);
                    report("New Table value set! Returns table"); return this.table.get(name).get(key);
                };
            }else {
                report("Table not found! Returns undefined"); return this.table.get(name).set(key, value);
            };
        };
    };
    /**
     * Deletes a table
     * @param {string} name The table to delete
     * @returns {undefined|Map}
     */
    deleteTable(name){
        if(name == ""){
            report("Given \"\" as table name! Returns undefined"); return undefined;
        }else {
            if(this.table.has(name)){
                let ov = this.table.get(name); this.table.delete(name); report("Succesfully deleted Table! Returns "+name); return ov;
            }else {
                report("Table to delete not found! Returns undefined"); return undefined;
            };
        };
    };
    /**
     * Deletes a value (key) in a given table
     * @param {string} name The table the key is located in
     * @param {string} key The key to delete
     * @returns {undefined|void}
     */
    deleteInTable(name, key){
        if(name == ""){
            report("Given \"\" as table name! Returns undefined"); return undefined;
        }else {
            if(this.table.has(name)){
                this.table.get(name).delete(key);
            }else {
                report("Given key does not exist inn table! Returns undefined"); return undefined;
            };
        };
    };
    /**
     * Loads the data from multiple Dynamic Properties of an object
     * @param {World|Entity} object The object to grab the data from
     * @param {string} baseProperty The base name of the properties (eg. name__1, name__2)
     */
    loadFromPropertyIds(object, baseProperty){
        let ids = object.getDynamicPropertyIds();
        let r = "";
        for(let i = 0; i < ids.length; i++){
            ids.sort();
            if(ids[i].startsWith(baseProperty)){
                r + object.getDynamicProperty(ids[i]);
            };
            
        }
        if(r == ""){
            report("No property data found! Check for misspelling"); return undefined;
        };this.loadSaveString(r);
    };
    /**
     * Loads the data from a saved Json table string
     * @param {string} tables A JSON saved Map in string
     */
    loadSaveString(tables) {
        let r = JSON.parse(tables);
        for(let key = 0; key < r.length; key++){
            this.table.set(r[key][0], new Map(Object.entries(r[key][1])));
        };
        report("Loaded Database set")
        return this.table;
    };
    /**
     * Returns a JSON string of the table data
     * @returns {string} A string that can be loaded with Database.loadSaveString()
     */
    getSaveString() {
        let vals = [];
        this.table.forEach((val)=>{
            let jsonF = Object.fromEntries(val);
            //report(jsonF); 	    // {k:v}
            vals.push(jsonF);
        });
        let keys = Array.from(this.table.keys());
        let r = [];
        for(let k = 0; k < keys.length; k++){
            let pair = [keys[k], vals[k]];
            //report(pair);
            r.push(pair);
        };
        return JSON.stringify(r);
    };
    /**
     * Gets a key's value in a table
     * @param {string} name The table to look in for
     * @param {string} key The key in the table to look for
     * @returns {undefined|any}
     */
    getInTable(name, key){
        if(this.table.has(name)){
            if(this.table.get(name).has(key)){
                return this.table.get(name).get(key);
            }else {
                report("Key not found! Returns undefined"); return undefined;
            };
        }else {
            report("Table not found! Returns undefined, searched "+name); return undefined;
        };
    };
    /**
     * Saves the Database inside an object's dynamic property
     * @param {string} property The dynamic property's identifier
     * @param {World|Entity} object The object to save in
     */
    saveDatabase(property, object){
        let size = getByteSize(this.getSaveString());
        if(size>32000){
            let chunk = splitStringIntoChunks(this.getSaveString(), size);
            for(let p = 0; p < chunk.length; p++){
                object.setDynamicProperty(property+"__"+p, chunk[p]);
            };
        }else {
            object.setDynamicProperty(property, this.getSaveString());
        }
        if(object instanceof World){
            report("Saved Database in World Dynamic Properties as "+property);
            report("Size function: "+getByteSize(this.getSaveString()));
            
        }else {
            report("Saved Database in Entity with Id "+ object.id+" as "+ property);
            report("Size function: "+getByteSize(this.getSaveString()));
        };
    };
    /**
     * @type {Map<string, Map<string, string|number|Array>>}
     */
    table = new Map();
};
/**
 * Get the byte size of a string.
 * @param {string} str - The input string.
 * @returns {number} - The byte size of the string.
 */
function getByteSize(str) {
    let byteSize = 0;

    for (let i = 0; i < str.length; i++) {
        const charCode = str.charCodeAt(i);

        // Count each byte based on UTF-8 encoding rules
        if (charCode <= 0x7F) {
            byteSize += 1;
        } else if (charCode <= 0x7FF) {
            byteSize += 2;
        } else if (charCode <= 0xFFFF) {
            byteSize += 3;
        } else {
            byteSize += 4;
        }
    }

    return byteSize;
}
/**
 * Split a string into chunks with a maximum byte size.
 * @param {string} str - The input string.
 * @param {number} chunkSize - The maximum byte size for each chunk.
 * @returns {string[]} - An array of string chunks.
 */
function splitStringIntoChunks(str, chunkSize) {
    const chunks = [];
    let currentChunk = '';

    for (let i = 0; i < str.length; i++) {
        const char = str[i];
        const charCode = char.charCodeAt(0);

        if ((currentChunk.length + char.length) * 2 > chunkSize) {
            chunks.push(currentChunk);
            currentChunk = '';
        }

        currentChunk += char;
    }

    if (currentChunk.length > 0) {
        chunks.push(currentChunk);
    }

    return chunks;
}